<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Leagues;
use App\Models\Team;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class LeaguesController extends Controller
{
    /**
     * Display a listing of all leagues
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLeagues(Request $request)
    {
        // Cache duration in minutes
        $cacheDuration = 60 * 24; // 24 hours
        
        // Get parameters from request
        $page = $request->input('page', 1);
        $perPage = $request->input('per_page', 25);
        $activeOnly = $request->input('active', true);
        $countryId = $request->input('country_id');
        $sportId = $request->input('sport_id');
        $searchTerm = $request->input('search');
        
        // Generate unique cache key based on parameters
        $cacheKey = "leagues:{$page}:{$perPage}:{$activeOnly}:{$countryId}:{$sportId}:{$searchTerm}";
        
        $leagues = Cache::remember($cacheKey, $cacheDuration, function () use ($activeOnly, $countryId, $sportId, $searchTerm, $perPage) {
            $query = Leagues::query();
            
            if ($activeOnly) {
                $query->where('active', true);
            }
            
            if ($countryId) {
                $query->where('country_id', $countryId);
            }
            
            if ($sportId) {
                $query->where('sport_id', $sportId);
            }
            
            if ($searchTerm) {
                $query->where(function($q) use ($searchTerm) {
                    $q->where('name', 'LIKE', "%{$searchTerm}%")
                      ->orWhere('short_code', 'LIKE', "%{$searchTerm}%");
                });
            }
            
            return $query->orderBy('name')->paginate($perPage);
        });
        
        return response()->json([
            'success' => true,
            'data' => $leagues,
            'message' => 'Leagues retrieved successfully'
        ]);
    }

    /**
     * Display the specified league
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLeague($id)
    {
        $cacheKey = "league:{$id}";
        $league = Cache::remember($cacheKey, 60 * 24, function () use ($id) {
            return Leagues::where('sportmonks_id', $id)
                ->orWhere('id', $id)
                ->first();
        });
        
        if (!$league) {
            return response()->json([
                'success' => false,
                'message' => 'League not found'
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'data' => $league,
            'message' => 'League retrieved successfully'
        ]);
    }

    /**
     * Get leagues by country
     *
     * @param int $countryId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLeaguesByCountry($countryId)
    {
        $cacheKey = "leagues:country:{$countryId}";
        $leagues = Cache::remember($cacheKey, 60 * 24, function () use ($countryId) {
            return Leagues::where('country_id', $countryId)
                ->where('active', true)
                ->orderBy('name')
                ->get();
        });
        
        return response()->json([
            'success' => true,
            'data' => $leagues,
            'message' => 'Leagues by country retrieved successfully'
        ]);
    }

    /**
     * Get live leagues
     */
    public function getLiveLeagues(Request $request)
    {
        try {
            $leagues = Leagues::where('active', true)
                ->whereNotNull('last_played_at')
                ->where('last_played_at', '>=', Carbon::now()->subHours(24))
                ->get();

            return response()->json([
                'success' => true,
                'data' => $leagues,
                'count' => $leagues->count()
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching live leagues: ' . $e->getMessage()
            ], 500);
        }
    }

     /**
     * Get leagues by fixture date
     */
    public function getLeaguesByFixtureDate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'date' => 'required|date_format:Y-m-d'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $date = Carbon::parse($request->input('date'));
            $leagues = Leagues::where('active', true)
                ->whereDate('last_played_at', $date)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $leagues,
                'count' => $leagues->count()
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching leagues by date: ' . $e->getMessage()
            ], 500);
        }
    }

     /**
     * Search leagues by name
     */
    public function searchLeaguesByName(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|min:2'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $leagues = Leagues::where('name', 'LIKE', '%' . $request->input('name') . '%')
                ->where('active', true)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $leagues,
                'count' => $leagues->count()
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error searching leagues: ' . $e->getMessage()
            ], 500);
        }
    }


     /* * Get all leagues by team ID via SportMonks API
     */
/**
     * Get all leagues by team ID
     */
public function getAllLeaguesByTeamId(Request $request, $team_id)
{
    // Validate team_id
    $validator = Validator::make(['team_id' => $team_id], [
        'team_id' => 'required|integer'
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'message' => 'Validation failed',
            'errors' => $validator->errors()
        ], 422);
    }

    try {
        // Check if team exists in local DB
        $team = Team::where('sportmonks_id', $team_id)->first();
        if (!$team) {
            return response()->json([
                'success' => false,
                'message' => 'Team not found'
            ], 404);
        }

        // Get API token
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            return response()->json([
                'success' => false,
                'message' => 'SportMonks API token not configured'
            ], 500);
        }

        // Call correct SportMonks API endpoint
        $response = Http::timeout(30)->get("https://api.sportmonks.com/v3/football/teams/{$team_id}", [
            'api_token' => $token,
            'include' => 'leagues;country'
        ]);

        if ($response->failed()) {
            return response()->json([
                'success' => false,
                'message' => "API request failed with status {$response->status()}: {$response->reason()}"
            ], 500);
        }

        $teamData = $response->json()['data'] ?? null;
        $leaguesData = $teamData['leagues'] ?? [];

        if (empty($leaguesData)) {
            return response()->json([
                'success' => true,
                'data' => [],
                'count' => 0
            ], 200);
        }

        $leagues = collect($leaguesData)->map(function ($league) {
            return [
                'sportmonks_id' => $league['id'] ?? null,
                'name' => $league['name'] ?? 'Unknown',
                'sport_id' => $league['sport_id'] ?? null,
                'country_id' => $league['country_id'] ?? null,
                'active' => $league['active'] ?? true,
                'short_code' => $league['short_code'] ?? null,
                'image_path' => $league['image_path'] ?? null,
                'type' => $league['type'] ?? null,
                'sub_type' => $league['sub_type'] ?? null,
                'last_played_at' => $league['last_played_at'] ?? null,
                'category' => $league['category'] ?? null,
                'has_jerseys' => $league['has_jerseys'] ?? false,
                'details' => $league
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $leagues,
            'count' => $leagues->count()
        ], 200);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error fetching leagues by team: ' . $e->getMessage()
        ], 500);
    }
}



    /**
     * Get current leagues by team ID
     */
    public function getCurrentLeaguesByTeamId(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'team_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Verify team exists in local database
            $team = Team::where('sportmonks_id', $request->input('team_id'))->first();
            if (!$team) {
                return response()->json([
                    'success' => false,
                    'message' => 'Team not found'
                ], 404);
            }

            $token = env('SPORTMONKS_TOKEN');
            if (empty($token)) {
                return response()->json([
                    'success' => false,
                    'message' => 'SportMonks API token not configured'
                ], 500);
            }

            $response = Http::timeout(30)->get("https://api.sportmonks.com/v3/football/teams/{$request->input('team_id')}/leagues", [
                'api_token' => $token,
                'include' => 'country',
                'filters' => 'leagueActive:1'
            ]);

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => "API request failed with status {$response->status()}: {$response->reason()}"
                ], 500);
            }

            $data = $response->json()['data'] ?? [];
            if (empty($data)) {
                return response()->json([
                    'success' => true,
                    'data' => [],
                    'count' => 0
                ], 200);
            }

            $leagues = collect($data)->map(function ($league) {
                return [
                    'sportmonks_id' => $league['id'] ?? null,
                    'name' => $league['name'] ?? 'Unknown',
                    'sport_id' => $league['sport_id'] ?? null,
                    'country_id' => $league['country']['id'] ?? $league['country_id'] ?? null,
                    'active' => $league['active'] ?? true,
                    'short_code' => $league['short_code'] ?? null,
                    'image_path' => $league['image_path'] ?? null,
                    'type' => $league['type'] ?? null,
                    'sub_type' => $league['sub_type'] ?? null,
                    'last_played_at' => $league['last_played_at'] ?? null,
                    'category' => $league['category'] ?? null,
                    'has_jerseys' => $league['has_jerseys'] ?? false,
                    'details' => $league
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $leagues,
                'count' => $leagues->count()
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching current leagues by team: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get shirts by league ID
     */
    public function getShirtsByLeagueId(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'league_id' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $league = Leagues::where('id', $request->input('league_id'))
                ->where('has_jerseys', true)
                ->first();

            if (!$league) {
                return response()->json([
                    'success' => false,
                    'message' => 'League not found or no jerseys available'
                ], 404);
            }

            $token = env('SPORTMONKS_TOKEN');
            if (empty($token)) {
                return response()->json([
                    'success' => false,
                    'message' => 'SportMonks API token not configured'
                ], 500);
            }

            $response = Http::timeout(30)->get("https://api.sportmonks.com/v3/football/leagues/{$league->sportmonks_id}", [
                'api_token' => $token,
                'include' => 'jerseys'
            ]);

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => "API request failed with status {$response->status()}: {$response->reason()}"
                ], 500);
            }

            $data = $response->json()['data']['jerseys'] ?? [];
            return response()->json([
                'success' => true,
                'data' => $data,
                'count' => count($data)
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching shirts: ' . $e->getMessage()
            ], 500);
        }
    }


}