<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Fixture;
use Illuminate\Http\Request;

class FixturesController extends Controller
{
      // Helper method to apply common relations
    private function withRelations($query)
    {
        return $query->with(['league', 'homeTeam', 'awayTeam']);
    }
    /**
     * Get all fixtures with optional filters
     */
    public function getFixtures(Request $request)
    {
        $query = Fixture::query()
            ->with(['league', 'homeTeam', 'awayTeam'])
            ->orderBy('starting_at');

        // Apply filters
        if ($request->has('league_id')) {
            $query->where('league_id', $request->league_id);
        }

        if ($request->has('team_id')) {
            $query->where(function($q) use ($request) {
                $q->where('home_team_id', $request->team_id)
                  ->orWhere('away_team_id', $request->team_id);
            });
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('date_from')) {
            $query->where('starting_at', '>=', $request->date_from);
        }

        if ($request->has('date_to')) {
            $query->where('starting_at', '<=', $request->date_to);
        }

        if ($request->has('has_odds')) {
            $query->where('has_odds', $request->has_odds);
        }

        // Pagination
        $perPage = $request->per_page ?? 20;
        $fixtures = $query->paginate($perPage);

        return response()->json([
            'data' => $fixtures->items(),
            'pagination' => [
                'total' => $fixtures->total(),
                'per_page' => $fixtures->perPage(),
                'current_page' => $fixtures->currentPage(),
                'last_page' => $fixtures->lastPage(),
            ]
        ]);
    }

      /**
     * Get a single fixture by ID
     */
    public function getFixture($id)
    {
        $fixture = Fixture::with(['league', 'homeTeam', 'awayTeam'])
            ->findOrFail($id);

        return response()->json($fixture);
    }

     /**
     * Get fixtures by league ID
     */
    public function getFixturesByLeague($leagueId)
    {
        $fixtures = Fixture::where('league_id', $leagueId)
            ->with(['league', 'homeTeam', 'awayTeam'])
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }

     /**
     * Get fixtures by team ID
     */
    public function getFixturesByTeam($teamId)
    {
        $fixtures = Fixture::where(function($query) use ($teamId) {
                $query->where('home_team_id', $teamId)
                      ->orWhere('away_team_id', $teamId);
            })
            ->with(['league', 'homeTeam', 'awayTeam'])
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }

     /**
     * Get live fixtures
     */
    public function getLiveFixtures()
    {
        $fixtures = Fixture::where('status', 'LIVE')
            ->with(['league', 'homeTeam', 'awayTeam'])
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }


     /**
     * Get upcoming fixtures
     */
    public function getUpcomingFixtures()
    {
        $fixtures = Fixture::where('status', 'SCHEDULED')
            ->where('starting_at', '>=', now())
            ->with(['league', 'homeTeam', 'awayTeam'])
            ->orderBy('starting_at')
            ->take(20) // Limit to 20 upcoming fixtures
            ->get();

        return response()->json($fixtures);
    }

     /**
     * Get fixtures by multiple IDs
     */
    public function getFixturesByIds(Request $request)
    {
        $ids = explode(',', $request->ids);
        $fixtures = $this->withRelations(Fixture::query())
            ->whereIn('id', $ids)
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }

      /**
     * Get fixtures by date range
     */
    public function getFixturesByDateRange(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $fixtures = $this->withRelations(Fixture::query())
            ->whereBetween('starting_at', [$request->start_date, $request->end_date])
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }

        /**
     * Get head-to-head fixtures between two teams
     */
    public function getHeadToHeadFixtures($team1Id, $team2Id)
    {
        $fixtures = $this->withRelations(Fixture::query())
            ->where(function($query) use ($team1Id, $team2Id) {
                $query->where('home_team_id', $team1Id)
                      ->where('away_team_id', $team2Id);
            })
            ->orWhere(function($query) use ($team1Id, $team2Id) {
                $query->where('home_team_id', $team2Id)
                      ->where('away_team_id', $team1Id);
            })
            ->orderBy('starting_at', 'desc')
            ->get();

        return response()->json($fixtures);
    }

       /**
     * Get latest updated fixtures
     */
    public function getLatestUpdatedFixtures()
    {
        $fixtures = $this->withRelations(Fixture::query())
            ->orderBy('updated_at', 'desc')
            ->take(20)
            ->get();

        return response()->json($fixtures);
    }

     /**
     * Get fixtures by date range for a specific team
     */
    public function getFixturesByDateRangeForTeam(Request $request, $teamId)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $fixtures = $this->withRelations(Fixture::query())
            ->where(function($query) use ($teamId) {
                $query->where('home_team_id', $teamId)
                      ->orWhere('away_team_id', $teamId);
            })
            ->whereBetween('starting_at', [$request->start_date, $request->end_date])
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }

      /**
     * Search fixtures by name
     */
    public function searchFixturesByName(Request $request)
    {
        $request->validate(['query' => 'required|string|min:3']);

        $fixtures = $this->withRelations(Fixture::query())
            ->where('name', 'like', '%' . $request->query . '%')
            ->orderBy('starting_at')
            ->get();

        return response()->json($fixtures);
    }
        public function getUpcomingFixturesByMarketId($marketId)
    {
        $fixtures = $this->withRelations(Fixture::query())
            ->where('status', 'SCHEDULED')
            ->where('starting_at', '>=', now())
            ->whereJsonContains('details->markets', (int)$marketId)
            ->orderBy('starting_at')
            ->take(20)
            ->get();

        return response()->json($fixtures);
    }

    
    /**
     * Get upcoming fixtures by TV station ID (from details JSON)
     */
    public function getUpcomingFixturesByTvStationId($tvStationId)
    {
        $fixtures = $this->withRelations(Fixture::query())
            ->where('status', 'SCHEDULED')
            ->where('starting_at', '>=', now())
            ->whereJsonContains('details->tvstations', (int)$tvStationId)
            ->orderBy('starting_at')
            ->take(20)
            ->get();

        return response()->json($fixtures);
    }

    
    /**
     * Get past fixtures by TV station ID (from details JSON)
     */
    public function getPastFixturesByTvStationId($tvStationId)
    {
        $fixtures = $this->withRelations(Fixture::query())
            ->where('status', 'FINISHED')
            ->where('starting_at', '<=', now())
            ->whereJsonContains('details->tvstations', (int)$tvStationId)
            ->orderBy('starting_at', 'desc')
            ->take(20)
            ->get();

        return response()->json($fixtures);
    }
    
}
