<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\TopScorer;
use App\Models\Topscorers;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksTopScorers extends Command
{
    protected $signature = 'fetch:sportmonks:topscorers';
    protected $description = 'Fetch and store top scorers data from SportMonks API';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        try {
            $response = Http::timeout(30)->get('https://api.sportmonks.com/v3/football/topscorers', [
                'api_token' => $token,
                'include' => 'player;league'
            ]);

            if ($response->failed()) {
                $this->logError("API request failed with status {$response->status()}: {$response->reason()}", $response->body());
                return 1;
            }

            $data = $response->json();
            if (empty($data)) {
                $this->logError('Empty API response');
                return 1;
            }

            if (!isset($data['data'])) {
                $this->logError('Invalid API response format - missing data key', json_encode($data));
                return 1;
            }

            $processed = 0;
            foreach ($data['data'] as $scorer) {
                try {
                    Topscorers::updateOrCreate(
                        [
                            'league_id' => $scorer['league_id'] ?? null,
                            'player_id' => $scorer['player']['id'] ?? null,
                        ],
                        [
                            'goals' => $scorer['goals'] ?? 0,
                            'assists' => $scorer['assists'] ?? null,
                            'details' => $scorer,
                        ]
                    );
                    $processed++;
                } catch (\Exception $e) {
                    $this->logError('Error processing top scorer for player ID ' . ($scorer['player']['id'] ?? 'unknown') . ': ' . $e->getMessage());
                }
            }

            $successMessage = "Successfully processed {$processed} top scorers at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
        } catch (\Exception $e) {
            $this->logError('SportMonks top scorers fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}