<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\Team;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksTeams extends Command
{
    protected $signature = 'fetch:sportmonks:teams';
    protected $description = 'Fetch and store all team data from SportMonks API without pagination limits';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        $baseUrl = 'https://api.sportmonks.com/v3/football/teams';
        $params = [
            'api_token' => $token,
            'include' => 'country;venue', // Include related data if needed
            'per_page' => 100, // Maximum per page allowed by API
        ];
        
        $page = 1;
        $totalProcessed = 0;
        $maxRequests = 1000; // Absolute safeguard against infinite loops

        try {
            do {
                $this->info("Fetching page {$page}");
                
                $response = Http::timeout(60)
                    ->retry(3, 1000)
                    ->get($baseUrl, array_merge($params, ['page' => $page]));

                if ($response->failed()) {
                    $this->logError(
                        "API request failed with status {$response->status()}: {$response->reason()}",
                        ['url' => $baseUrl, 'params' => $params, 'response' => $response->body()]
                    );
                    return 1;
                }

                $data = $response->json();
                if (empty($data)) {
                    $this->logError('Empty API response');
                    return 1;
                }

                if (!isset($data['data'])) {
                    $this->logError('Invalid API response format - missing data key', json_encode($data));
                    return 1;
                }

                $teams = $data['data'];
                if (empty($teams)) {
                    $this->info("No more teams found - ending process");
                    break;
                }

                $processed = $this->processTeamsBatch($teams);
                $totalProcessed += $processed;
                $this->info("Processed {$processed} teams on page {$page} (Total: {$totalProcessed})");

                // Check pagination
                $pagination = $data['pagination'] ?? [];
                $hasMore = $pagination['has_more'] ?? false;
                $page++;

                if (!$hasMore || $page > $maxRequests) {
                    if ($page > $maxRequests) {
                        $this->logWarning("Reached maximum request limit ({$maxRequests})");
                    }
                    break;
                }

                // Optional: Add delay between requests to avoid rate limiting
                if ($page % 10 === 0) {
                    sleep(1);
                }

            } while (true);

            $successMessage = "Successfully processed {$totalProcessed} teams at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
            
        } catch (\Exception $e) {
            $this->logError('SportMonks teams fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function processTeamsBatch(array $teams): int
    {
        $processed = 0;
        
        foreach ($teams as $team) {
            try {
                Team::updateOrCreate(
                    ['sportmonks_id' => $team['id']],
                    [
                        'name' => $team['name'] ?? 'Unknown',
                        'sport_id' => $team['sport_id'] ?? null,
                        'country_id' => $team['country_id'] ?? null,
                        'venue_id' => $team['venue_id'] ?? null,
                        'gender' => $team['gender'] ?? null,
                        'short_code' => $team['short_code'] ?? null,
                        'image_path' => $team['image_path'] ?? null,
                        'founded' => $team['founded'] ?? null,
                        'type' => $team['type'] ?? null,
                        'placeholder' => $team['placeholder'] ?? false,
                        'last_played_at' => $team['last_played_at'] ?? null,
                        'national_team' => $team['national_team'] ?? false,
                        'trophies' => null,
                        'transfers' => null,
                        'squads' => null,
                        'stats' => null,
                    ]
                );
                $processed++;
            } catch (\Exception $e) {
                $this->logError('Error processing team ID ' . ($team['id'] ?? 'unknown') . ': ' . $e->getMessage());
            }
        }
        
        return $processed;
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}