<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\Standing;
use App\Models\Standings;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksStandings extends Command
{
    protected $signature = 'fetch:sportmonks:standings';
    protected $description = 'Fetch and store standings data from SportMonks API';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        try {
            $response = Http::timeout(30)->get('https://api.sportmonks.com/v3/football/standings', [
                'api_token' => $token,
                'include' => 'team;league'
            ]);

            if ($response->failed()) {
                $this->logError("API request failed with status {$response->status()}: {$response->reason()}", $response->body());
                return 1;
            }

            $data = $response->json();
            if (empty($data)) {
                $this->logError('Empty API response');
                return 1;
            }

            if (!isset($data['data'])) {
                $this->logError('Invalid API response format - missing data key', json_encode($data));
                return 1;
            }

            $processed = 0;
            foreach ($data['data'] as $standing) {
                try {
                    Standings::updateOrCreate(
                        [
                            'league_id' => $standing['league_id'] ?? null,
                            'team_id' => $standing['team']['id'] ?? null,
                        ],
                        [
                            'position' => $standing['position'] ?? 0,
                            'points' => $standing['points'] ?? 0,
                            'played' => $standing['played'] ?? 0,
                            'won' => $standing['won'] ?? 0,
                            'drawn' => $standing['draw'] ?? 0,
                            'lost' => $standing['lost'] ?? 0,
                            'goals_for' => $standing['goals_for'] ?? 0,
                            'goals_against' => $standing['goals_against'] ?? 0,
                            'details' => $standing,
                        ]
                    );
                    $processed++;
                } catch (\Exception $e) {
                    $this->logError('Error processing standing for team ID ' . ($standing['team']['id'] ?? 'unknown') . ': ' . $e->getMessage());
                }
            }

            $successMessage = "Successfully processed {$processed} standings at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
        } catch (\Exception $e) {
            $this->logError('SportMonks standings fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}