<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\Prediction;
use App\Models\Predictions;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksPredictions extends Command
{
    protected $signature = 'fetch:sportmonks:predictions';
    protected $description = 'Fetch and store predictions data from SportMonks API';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        try {
            $response = Http::timeout(30)->get('https://api.sportmonks.com/v3/football/predictions', [
                'api_token' => $token,
                'include' => 'match'
            ]);

            if ($response->failed()) {
                $this->logError("API request failed with status {$response->status()}: {$response->reason()}", $response->body());
                return 1;
            }

            $data = $response->json();
            if (empty($data)) {
                $this->logError('Empty API response');
                return 1;
            }

            if (!isset($data['data'])) {
                $this->logError('Invalid API response format - missing data key', json_encode($data));
                return 1;
            }

            $processed = 0;
            foreach ($data['data'] as $prediction) {
                try {
                    Predictions::updateOrCreate(
                        [
                            'match_id' => $prediction['match_id'] ?? null,
                            'prediction_type' => $prediction['type'] ?? 'unknown',
                        ],
                        [
                            'value' => $prediction['value'] ?? 'unknown',
                            'probability' => $prediction['probability'] ?? null,
                        ]
                    );
                    $processed++;
                } catch (\Exception $e) {
                    $this->logError('Error processing prediction for match ID ' . ($prediction['match_id'] ?? 'unknown') . ': ' . $e->getMessage());
                }
            }

            $successMessage = "Successfully processed {$processed} predictions at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
        } catch (\Exception $e) {
            $this->logError('SportMonks predictions fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}