<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\Player;
use App\Models\Players;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksPlayers extends Command
{
    protected $signature = 'fetch:sportmonks:players';
    protected $description = 'Fetch and store player data from SportMonks API';

    // Add a maximum page limit to prevent infinite loops
    protected $maxPages = 100;
    protected $currentPage = 0;

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        $baseUrl = 'https://api.sportmonks.com/v3/football/players';
        $params = [
            'api_token' => $token,
            'per_page' => 25, // Explicitly set per_page to match your batch size
            // 'include' => 'team;statistics;injuries' // Uncomment if needed
        ];

        try {
            do {
                $this->currentPage++;
                
                // Add page parameter
                $params['page'] = $this->currentPage;

                $this->info("Fetching page {$this->currentPage}...");
                
                $response = Http::timeout(60)->get($baseUrl, $params);

                if ($response->failed()) {
                    $this->logError("API request failed with status {$response->status()}: {$response->reason()}", $response->body());
                    return 1;
                }

                $data = $response->json();
                
                if (empty($data)) {
                    $this->logError('Empty API response');
                    return 1;
                }

                if (!isset($data['data'])) {
                    $this->logError('Invalid API response format - missing data key', json_encode($data));
                    return 1;
                }

                $players = $data['data'];
                if (empty($players)) {
                    $this->info("No more players found. Ending process.");
                    break;
                }

                $processed = 0;
                foreach ($players as $player) {
                    try {
                        Players::updateOrCreate(
                            ['sportmonks_id' => $player['id']],
                            [
                                'name' => $player['name'] ?? $player['display_name'] ?? 'Unknown',
                                'common_name' => $player['common_name'] ?? null,
                                'firstname' => $player['firstname'] ?? null,
                                'lastname' => $player['lastname'] ?? null,
                                'display_name' => $player['display_name'] ?? null,
                                'team_id' => $player['team']['id'] ?? null,
                                'country_id' => $player['country_id'] ?? null,
                                'position_id' => $player['position_id'] ?? null,
                                'detailed_position_id' => $player['detailed_position_id'] ?? null,
                                'type_id' => $player['type_id'] ?? null,
                                'image_path' => $player['image_path'] ?? null,
                                'height' => $player['height'] ?? null,
                                'weight' => $player['weight'] ?? null,
                                'date_of_birth' => $player['date_of_birth'] ?? null,
                                'gender' => $player['gender'] ?? null,
                                'stats' => $player['statistics'] ?? null,
                                'injuries' => $player['injuries'] ?? null,
                            ]
                        );
                        $processed++;
                    } catch (\Exception $e) {
                        $this->logError('Error processing player ID ' . ($player['id'] ?? 'unknown') . ': ' . $e->getMessage());
                        continue;
                    }
                }

                $this->info("Processed {$processed} players in this batch");

                // Check if we should continue pagination
                $pagination = $data['pagination'] ?? null;
                if (!$pagination || !isset($pagination['has_more']) || !$pagination['has_more']) {
                    break;
                }

                // Prevent infinite loops
                if ($this->currentPage >= $this->maxPages) {
                    $this->warn("Reached maximum page limit of {$this->maxPages}. Stopping.");
                    break;
                }

                // Small delay to avoid rate limiting
                sleep(1);

            } while (true);

            $successMessage = "Successfully processed players. Total pages: {$this->currentPage}";
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
        } catch (\Exception $e) {
            $this->logError('SportMonks players fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}