<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\League;
use App\Models\Leagues;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;

class FetchSportMonksLeagues extends Command
{
    protected $signature = 'fetch:sportmonks:leagues';
    protected $description = 'Fetch and store league data from SportMonks API';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        $url = 'https://api.sportmonks.com/v3/football/leagues';
        $params = [
            'api_token' => $token,
            'include' => 'country'
        ];
        $page = 1;
        $maxPages = 100; // Safeguard to prevent infinite loop
        $totalProcessed = 0;

        try {
            do {
                $this->info("Fetching page {$page}");
                $response = Http::timeout(30)->get($url, $params);

                if ($response->failed()) {
                    $this->logError("API request failed with status {$response->status()}: {$response->reason()}", $response->body());
                    return 1;
                }

                $data = $response->json();
                if (empty($data)) {
                    $this->logError('Empty API response');
                    return 1;
                }

                if (!isset($data['data'])) {
                    $this->logError('Invalid API response format - missing data key', json_encode($data));
                    return 1;
                }

                $leagues = $data['data'];
                if (empty($leagues)) {
                    $this->logWarning("No leagues found on page {$page}");
                    break;
                }

                $processed = 0;
                foreach ($leagues as $league) {
                    try {
                        Leagues::updateOrCreate(
                            ['sportmonks_id' => $league['id']],
                            [
                                'name' => $league['name'] ?? 'Unknown',
                                'sport_id' => $league['sport_id'] ?? null,
                                'country_id' => $league['country']['id'] ?? $league['country_id'] ?? null,
                                'active' => $league['active'] ?? true,
                                'short_code' => $league['short_code'] ?? null,
                                'image_path' => $league['image_path'] ?? null,
                                'type' => $league['type'] ?? null,
                                'sub_type' => $league['sub_type'] ?? null,
                                'last_played_at' => $league['last_played_at'] ?? null,
                                'category' => $league['category'] ?? null,
                                'has_jerseys' => $league['has_jerseys'] ?? false,
                                'details' => $league,
                            ]
                        );
                        $processed++;
                    } catch (\Exception $e) {
                        $this->logError('Error processing league ID ' . ($league['id'] ?? 'unknown') . ': ' . $e->getMessage());
                    }
                }

                $totalProcessed += $processed;
                $this->info("Processed {$processed} leagues on page {$page} (Total: {$totalProcessed})");

                // Log pagination for debugging
                $this->info('Pagination: ' . json_encode($data['pagination'] ?? []));

                // Check pagination
                $hasMore = $data['pagination']['has_more'] ?? false;
                $url = $data['pagination']['next_page'] ?? null;
                $params = ['api_token' => $token];
                $page++;

                // Safeguard: Stop if max pages reached
                if ($page > $maxPages) {
                    $this->logWarning("Reached maximum page limit ({$maxPages})");
                    break;
                }

                // Stop if no more data or no next page
                if (!$hasMore || empty($url)) {
                    break;
                }
            } while (true);

            $successMessage = "Successfully processed {$totalProcessed} leagues at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
        } catch (\Exception $e) {
            $this->logError('SportMonks leagues fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}