<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use App\Models\Fixture;
use App\Models\Team;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;

class FetchSportMonksFixtures extends Command
{
    protected $signature = 'fetch:sportmonks:fixtures';
    protected $description = 'Fetch and store all fixture data from SportMonks API without pagination limits';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');
        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        $baseUrl = 'https://api.sportmonks.com/v3/football/fixtures';
        $params = [
            'api_token' => $token,
            'include' => 'league;participants;venue;round;stage',
            'per_page' => 100, // Maximum per page allowed by API
        ];
        
        $page = 1;
        $totalProcessed = 0;
        $maxRequests = 1000; // Absolute safeguard against infinite loops

        // Map state_id to status
        $statusMap = [
            1 => 'SCHEDULED',
            2 => 'LIVE',
            3 => 'FINISHED',
            4 => 'CANCELLED',
            5 => 'POSTPONED',
            6 => 'SUSPENDED',
            7 => 'INTERRUPTED',
            8 => 'ABANDONED',
            9 => 'DELAYED',
            10 => 'TBD',
            11 => 'AWARDED',
        ];

        try {
            do {
                $this->info("Fetching page {$page}");
                
                $response = Http::timeout(60)
                    ->retry(3, 1000)
                    ->get($baseUrl, array_merge($params, ['page' => $page]));

                if ($response->failed()) {
                    $this->logError(
                        "API request failed with status {$response->status()}: {$response->reason()}",
                        ['url' => $baseUrl, 'params' => $params, 'response' => $response->body()]
                    );
                    return 1;
                }

                $data = $response->json();
                if (empty($data)) {
                    $this->logError('Empty API response');
                    return 1;
                }

                if (!isset($data['data'])) {
                    $this->logError('Invalid API response format - missing data key', json_encode($data));
                    return 1;
                }

                $fixtures = $data['data'];
                if (empty($fixtures)) {
                    $this->info("No more fixtures found - ending process");
                    break;
                }

                $processed = $this->processFixturesBatch($fixtures, $statusMap);
                $totalProcessed += $processed;
                $this->info("Processed {$processed} fixtures on page {$page} (Total: {$totalProcessed})");

                // Check pagination
                $pagination = $data['pagination'] ?? [];
                $hasMore = $pagination['has_more'] ?? false;
                $page++;

                if (!$hasMore || $page > $maxRequests) {
                    if ($page > $maxRequests) {
                        $this->logWarning("Reached maximum request limit ({$maxRequests})");
                    }
                    break;
                }

                // Add delay between requests to avoid rate limiting
                if ($page % 5 === 0) {
                    sleep(1);
                }

            } while (true);

            $successMessage = "Successfully processed {$totalProcessed} fixtures at " . now();
            $this->info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            return 0;
            
        } catch (\Exception $e) {
            $this->logError('SportMonks fixtures fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function processFixturesBatch(array $fixtures, array $statusMap): int
    {
        $processed = 0;
        
        foreach ($fixtures as $fixture) {
            try {
                // Get team IDs from participants if available
                $homeTeamId = null;
                $awayTeamId = null;
                $participants = $fixture['participants'] ?? [];

                if (count($participants) >= 2) {
                    $homeParticipant = $participants[0];
                    $awayParticipant = $participants[1];
                    
                    if ($homeParticipant['meta']['location'] === 'home') {
                        $homeTeamId = $homeParticipant['id'];
                        $awayTeamId = $awayParticipant['id'];
                    } else {
                        $homeTeamId = $awayParticipant['id'];
                        $awayTeamId = $homeParticipant['id'];
                    }
                } else {
                    // Fallback to name parsing if participants not available
                    list($homeTeamId, $awayTeamId) = $this->getTeamIdsFromFixtureName($fixture);
                }

                // Skip if both teams are null (placeholder fixture)
                if ($homeTeamId === null && $awayTeamId === null && 
                    ($fixture['placeholder'] ?? false)) {
                    $this->info("Skipping placeholder fixture ID: {$fixture['id']}");
                    continue;
                }

                // Prepare scores if available
                $scores = $fixture['scores'] ?? [];
                $scoreData = $this->extractScoreData($scores);

                Fixture::updateOrCreate(
                    ['sportmonks_id' => $fixture['id']],
                    [
                        'league_id' => $fixture['league_id'] ?? null,
                        'home_team_id' => $homeTeamId,
                        'away_team_id' => $awayTeamId,
                        'sport_id' => $fixture['sport_id'] ?? null,
                        'season_id' => $fixture['season_id'] ?? null,
                        'stage_id' => $fixture['stage_id'] ?? null,
                        'group_id' => $fixture['group_id'] ?? null,
                        'round_id' => $fixture['round_id'] ?? null,
                        'venue_id' => $fixture['venue_id'] ?? null,
                        'referee_id' => $fixture['referee_id'] ?? null,
                        'name' => $fixture['name'] ?? null,
                        'starting_at' => $fixture['starting_at'] ?? null,
                        'status' => $statusMap[$fixture['state_id'] ?? 1] ?? 'SCHEDULED',
                        'leg' => $fixture['leg'] ?? null,
                        'length' => $fixture['length'] ?? null,
                        'placeholder' => $fixture['placeholder'] ?? false,
                        'has_odds' => $fixture['has_odds'] ?? false,
                        'has_premium_odds' => $fixture['has_premium_odds'] ?? false,
                        'home_score' => $scoreData['home_score'] ?? null,
                        'away_score' => $scoreData['away_score'] ?? null,
                        'home_penalty' => $scoreData['home_penalty'] ?? null,
                        'away_penalty' => $scoreData['away_penalty'] ?? null,
                        'winner_id' => $scoreData['winner_id'] ?? null,
                        'details' => $fixture,
                    ]
                );
                $processed++;
            } catch (\Exception $e) {
                $this->logError('Error processing fixture ID ' . ($fixture['id'] ?? 'unknown') . ': ' . $e->getMessage());
            }
        }
        
        return $processed;
    }

    protected function getTeamIdsFromFixtureName(array $fixture): array
    {
        $homeTeamId = null;
        $awayTeamId = null;

        if (!empty($fixture['name'])) {
            $teams = explode(' vs ', $fixture['name']);
            if (count($teams) === 2) {
                $homeTeam = trim($teams[0]);
                $awayTeam = trim($teams[1]);

                // Skip placeholder teams
                if (Str::contains($homeTeam, ['Winner Match', '1st Group', '2nd Group', 'To Be Determined']) ||
                    Str::contains($awayTeam, ['Winner Match', '1st Group', '2nd Group', 'To Be Determined'])) {
                    return [null, null];
                }

                // Normalize team names (remove accents, lowercase)
                $homeTeamNormalized = Str::ascii(strtolower($homeTeam));
                $awayTeamNormalized = Str::ascii(strtolower($awayTeam));

                // Query teams with robust matching
                $homeTeamRecord = Team::whereRaw('LOWER(name) = ?', [$homeTeamNormalized])
                    ->orWhereRaw('LOWER(short_code) = ?', [$homeTeamNormalized])
                    ->orWhere('name', 'LIKE', "%{$homeTeam}%")
                    ->orWhere('short_code', 'LIKE', "%{$homeTeam}%")
                    ->first();
                $awayTeamRecord = Team::whereRaw('LOWER(name) = ?', [$awayTeamNormalized])
                    ->orWhereRaw('LOWER(short_code) = ?', [$awayTeamNormalized])
                    ->orWhere('name', 'LIKE', "%{$awayTeam}%")
                    ->orWhere('short_code', 'LIKE', "%{$awayTeam}%")
                    ->first();

                $homeTeamId = $homeTeamRecord->sportmonks_id ?? null;
                $awayTeamId = $awayTeamRecord->sportmonks_id ?? null;

                // Log unmatched teams
                if (!$homeTeamRecord) {
                    $this->logWarning("Home team not found for fixture ID {$fixture['id']}: {$homeTeam}");
                }
                if (!$awayTeamRecord) {
                    $this->logWarning("Away team not found for fixture ID {$fixture['id']}: {$awayTeam}");
                }
            }
        }

        return [$homeTeamId, $awayTeamId];
    }

    protected function extractScoreData(array $scores): array
    {
        $result = [
            'home_score' => null,
            'away_score' => null,
            'home_penalty' => null,
            'away_penalty' => null,
            'winner_id' => null,
        ];

        foreach ($scores as $score) {
            if ($score['description'] === 'CURRENT') {
                $result['home_score'] = $score['score']['home'] ?? null;
                $result['away_score'] = $score['score']['away'] ?? null;
            } elseif ($score['description'] === 'PENALTIES') {
                $result['home_penalty'] = $score['score']['home'] ?? null;
                $result['away_penalty'] = $score['score']['away'] ?? null;
            }
        }

        // Determine winner if scores exist
        if ($result['home_score'] !== null && $result['away_score'] !== null) {
            if ($result['home_score'] > $result['away_score']) {
                $result['winner_id'] = 'home';
            } elseif ($result['home_score'] < $result['away_score']) {
                $result['winner_id'] = 'away';
            } else {
                if ($result['home_penalty'] !== null && $result['away_penalty'] !== null) {
                    $result['winner_id'] = $result['home_penalty'] > $result['away_penalty'] ? 'home' : 'away';
                }
            }
        }

        return $result;
    }

    protected function logError($message, $context = null)
    {
        \Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message)
    {
        \Log::warning($message);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}