<?php

namespace App\Console\Commands;

use App\Models\DebugLog;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Matches;

class FetchSportMonksData extends Command
{
    protected $signature = 'fetch:sportmonks';
    protected $description = 'Fetch and store football data from Sportmonks API';

    public function handle()
    {
        $token = env('SPORTMONKS_TOKEN');

        if (empty($token)) {
            $this->logError('SportMonks token not found in .env file');
            return 1;
        }

        try {
            $response = Http::timeout(30)->get('https://api.sportmonks.com/v3/football/livescores', [
                'api_token' => $token,
                'include' => 'participants;league;scores'
            ]);

            // Log the full response for debugging
            Log::info('SportMonks API Response', ['response' => $response->json(), 'headers' => $response->headers()]);

            if ($response->failed()) {
                $this->logError('API request failed: ' . $response->reason(), $response->body());
                return 1;
            }

            $data = $response->json();

            if (!isset($data['data']) || empty($data['data'])) {
                $message = isset($data['error'])
                    ? 'API error: ' . json_encode($data['error'])
                    : 'No live matches found or invalid API response format';
                $this->logWarning($message, json_encode($data));
                return 0;
            }

            $processed = 0;

            foreach ($data['data'] as $match) {
                try {
                    $participants = $match['participants'] ?? [];

                    if (count($participants) < 2) {
                        $this->logWarning('Invalid match participants for match ID: ' . ($match['id'] ?? 'unknown'));
                        continue;
                    }

                    Matches::updateOrCreate(
                        ['sportmonks_id' => $match['id']],
                        [
                            'league_id' => $match['league_id'] ?? null,
                            'home_team_id' => $participants[0]['id'] ?? null,
                            'away_team_id' => $participants[1]['id'] ?? null,
                            'start_time' => isset($match['starting_at_timestamp'])
                                ? date('Y-m-d H:i:s', $match['starting_at_timestamp'])
                                : null,
                            'status' => $match['state'] ?? 'unknown',
                            'score' => $this->formatScore($match),
                            'events' => $match['events'] ?? null,
                            'tv_stations' => $match['tv_stations'] ?? null,
                            'commentaries' => $match['commentaries'] ?? null,
                            'stats' => $match['statistics'] ?? null,
                            'lineups' => $match['lineups'] ?? null,
                            'ball_coordinates' => $match['ball_coordinates'] ?? null,
                            'trends' => $match['trends'] ?? null,
                            'standings' => $match['standings'] ?? null,
                        ]
                    );
                    $processed++;
                } catch (\Exception $e) {
                    $this->logError('Error processing match ID ' . ($match['id'] ?? 'unknown') . ': ' . $e->getMessage());
                }
            }

            $successMessage = "Successfully processed {$processed} matches at " . now();
            Log::info($successMessage);
            DebugLog::create(['message' => $successMessage]);
            $this->info($successMessage);
            return 0;

        } catch (\Exception $e) {
            $this->logError('SportMonks fetch error: ' . $e->getMessage());
            return 1;
        }
    }

    protected function formatScore($match): ?string
    {
        $scores = $match['scores'] ?? null;

        if ($scores && isset($scores['localteam_score'], $scores['visitorteam_score'])) {
            return "{$scores['localteam_score']}-{$scores['visitorteam_score']}";
        }

        return null;
    }

    protected function logError($message, $context = null)
    {
        Log::error($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->error($message);
    }

    protected function logWarning($message, $context = null)
    {
        Log::warning($message, ['context' => $context]);
        DebugLog::create(['message' => $message]);
        $this->warn($message);
    }
}